// SPDX-FileCopyrightText: Copyright (C) 2025
// SPDX-License-Identifier: MPL-2.0

const socket = io(`http://${window.location.host}`);
let currentState = {
    direction: 'stop',
    speed: 150,
    last_update: null
};

// 初始化页面元素
document.addEventListener('DOMContentLoaded', () => {
    initializeUI();
    setupEventListeners();
    updateStatusDisplay();
    
    // 获取初始状态
    fetchStatus();
});

function initializeUI() {
    // 设置速度滑块
    const speedSlider = document.getElementById('speedSlider');
    const speedValue = document.getElementById('speedValue');
    
    speedSlider.value = currentState.speed;
    speedValue.textContent = currentState.speed;
    
    // 更新当前速度显示
    document.getElementById('currentSpeed').textContent = currentState.speed;
}

function setupEventListeners() {
    // 速度滑块事件
    const speedSlider = document.getElementById('speedSlider');
    speedSlider.addEventListener('input', handleSpeedChange);
    speedSlider.addEventListener('change', handleSpeedCommit);
    
    // 预设速度按钮
    document.querySelectorAll('.speed-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const speed = parseInt(this.getAttribute('data-speed'));
            setSpeed(speed);
        });
    });
    
    // 停止按钮
    const stopBtn = document.querySelector('.stop-btn');
    stopBtn.addEventListener('click', handleStop);
}

function handleSpeedChange(event) {
    const speed = parseInt(event.target.value);
    document.getElementById('speedValue').textContent = speed;
    document.getElementById('currentSpeed').textContent = speed;
}

function handleSpeedCommit(event) {
    const speed = parseInt(event.target.value);
    setSpeed(speed);
}

function setSpeed(speed) {
    socket.emit('set_speed', speed);
    document.getElementById('speedSlider').value = speed;
    document.getElementById('speedValue').textContent = speed;
    document.getElementById('currentSpeed').textContent = speed;
}

// 触摸事件处理
function handleTouchStart(event) {
    event.preventDefault();
    const button = event.currentTarget;
    const action = button.getAttribute('data-action');
    
    // 添加激活状态
    button.classList.add('active');
    
    // 发送触摸开始事件
    socket.emit('touch_start', {
        action: action,
        id: Date.now() // 唯一ID
    });
}

function handleTouchEnd(event) {
    event.preventDefault();
    const button = event.currentTarget;
    const action = button.getAttribute('data-action');
    
    // 移除激活状态
    button.classList.remove('active');
    
    // 发送触摸结束事件
    socket.emit('touch_end', {
        action: action,
        id: Date.now()
    });
}

// 鼠标事件处理（桌面端支持）
function handleMouseDown(event) {
    const button = event.currentTarget;
    const action = button.getAttribute('data-action');
    
    // 添加激活状态
    button.classList.add('active');
    
    // 发送触摸开始事件
    socket.emit('touch_start', {
        action: action,
        id: Date.now()
    });
}

function handleMouseUp(event) {
    const button = event.currentTarget;
    const action = button.getAttribute('data-action');
    
    // 移除激活状态
    button.classList.remove('active');
    
    // 发送触摸结束事件
    socket.emit('touch_end', {
        action: action,
        id: Date.now()
    });
}

function handleStop() {
    socket.emit('stop_motors');
}

// 获取状态
function fetchStatus() {
    fetch(`http://${window.location.host}/status`, { cache: "no-store" })
        .then(r => r.ok ? r.json() : Promise.reject(r.status))
        .then(data => {
            currentState = data;
            updateStatusDisplay();
        })
        .catch(err => {
            console.error('Failed to fetch status:', err);
        });
}

// 更新状态显示
function updateStatusDisplay() {
    const directionText = {
        'stop': '停止',
        'forward': '前进',
        'backward': '后退',
        'left': '左转',
        'right': '右转'
    };
    
    document.getElementById('currentStatus').textContent = 
        directionText[currentState.direction] || currentState.direction;
    
    document.getElementById('currentSpeed').textContent = currentState.speed;
    
    if (currentState.last_update) {
        const date = new Date(currentState.last_update);
        document.getElementById('lastUpdate').textContent = 
            `最后更新: ${date.toLocaleString()}`;
    }
    
    // 更新速度滑块
    const speedSlider = document.getElementById('speedSlider');
    const speedValue = document.getElementById('speedValue');
    
    if (speedSlider.value != currentState.speed) {
        speedSlider.value = currentState.speed;
        speedValue.textContent = currentState.speed;
    }
}

// Socket.IO 事件监听
socket.on('connect', () => {
    console.log('Connected to server');
    document.querySelector('.status-indicator').classList.add('connected');
});

socket.on('disconnect', () => {
    console.log('Disconnected from server');
    document.querySelector('.status-indicator').classList.remove('connected');
});

socket.on('car_state_update', (data) => {
    currentState = data;
    updateStatusDisplay();
});

socket.on('error', (message) => {
    console.error('Server error:', message);
    alert('错误: ' + message);
});

// 心跳检查
setInterval(fetchStatus, 5000);